import { inject } from "@angular/core"
import { Course } from "@core/domain-classes/course"
import { patchState, signalStore, withHooks, withMethods, withState } from "@ngrx/signals"
import { ToastrService } from '@core/services/toastr-service';
import { CourseService } from "../course.service"
import { TranslationService } from "@core/services/translation.service"
import { tapResponse } from "@ngrx/operators"
import { rxMethod } from "@ngrx/signals/rxjs-interop"
import { pipe, tap, switchMap } from "rxjs"
import { toObservable } from "@angular/core/rxjs-interop"
import { withStorageSync } from '@angular-architects/ngrx-toolkit';
import { CourseResource } from "@core/domain-classes/course-resource"

type CourseState = {
    courseList: Course[],
    currentCourse: Course | null,
    isAddorUpdate: boolean,
    loadList: boolean,
    filterParameter: CourseResource
}

const initialState: CourseState = {
    courseList: [],
    currentCourse: null,
    isAddorUpdate: false,
    loadList: true,
    filterParameter: {
        name: '',
        title: '',
        description: '',
        orderBy: 'title asc',
        skip: 0,
        pageSize: 10,
        totalCount: 0
    }
}

export const CourseStore = signalStore(
    { providedIn: 'root' },
    withStorageSync({
        key: 'courses',
        autoSync: true,
        storage: () => sessionStorage,
    }),
    withState(initialState),
    withMethods((
        store,
        courseService = inject(CourseService),
        toastr = inject(ToastrService),
        translationService = inject(TranslationService)
    ) => ({
        loadByQuery: rxMethod<CourseResource>(
            pipe(
                tap(() => patchState(store, { courseList: [] })),
                switchMap((filter) => {
                    return courseService.getCourses(filter).pipe(
                        tapResponse({
                            next: (httpResponse) => {
                                if (httpResponse && httpResponse.body) {
                                    patchState(store, { courseList: [...httpResponse.body], loadList: false })
                                }
                                if (httpResponse.headers.get('x-pagination')) {
                                    const pagination: CourseResource = JSON.parse(httpResponse.headers.get('x-pagination') ?? '');
                                    patchState(store, { filterParameter: { ...filter, totalCount: pagination.totalCount } })
                                }
                            },
                            error: (error: any) => {
                                toastr.error(error);
                            }
                        })
                    );
                })
            )
        ),
        addCourse(course: Course) {
            courseService.addCourse(course).subscribe({
                next: (res) => {
                    const course = res as Course
                    if (course) {
                        patchState(store, { currentCourse: { ...course }, isAddorUpdate: true, loadList: true });
                        toastr.success(translationService.getValue('COURSE_SAVE_SUCCESSFULLY'));
                    }
                },
                error: (err: any) => {
                    toastr.error(err.error);
                }
            });
        },
        updateCourse(updateCourse: Course) {
            courseService.updateCourse(updateCourse).subscribe({
                next: () => {
                    patchState(store, {
                        courseList: store.courseList().map((course) =>
                            updateCourse.id === course.id ? updateCourse : course
                        ),
                        isAddorUpdate: true,
                        currentCourse: { ...updateCourse }
                    });
                    toastr.success(translationService.getValue('COURSE_SAVE_SUCCESSFULLY'));
                },
                error: (err: any) => {
                    toastr.error(err.error);
                }
            })
        },
        deleteCourse(courseId: string) {
            courseService.deleteCourse(courseId).subscribe({
                next: () => {
                    patchState(store, { loadList: true });
                    toastr.success(translationService.getValue('COURSE_DELETE_SUCCESSFULLY'))
                }
            })
        },
        resetflag() {
            patchState(store, { isAddorUpdate: false })
        },
        setCurrentCouse(course: Course) {
            patchState(store, { currentCourse: { ...course } })
        }
    })),
    withHooks({
        onInit(store) {
            toObservable(store.loadList).subscribe((flag) => {
                if (flag) {
                    store.loadByQuery(store.filterParameter());
                }
            })
        }
    })
)
