import { HttpClient, HttpEvent } from '@angular/common/http';
import { inject, Injectable } from '@angular/core';
import { CourseSessionMedia, ResumeCourseSessionMedia } from '@core/domain-classes/course-session-media';
import { EmployeeCourse } from '@core/domain-classes/employee-course-session';
import { EmployeeCourseSessionActivity } from '@core/domain-classes/employee-course-session-activity';
import { EmployeeCourseSessionQuiz } from '@core/domain-classes/employee-course-session-quiz';
import { SessionQuiz } from '@core/domain-classes/session-quiz';
import { QuizResponse } from '@core/domain-classes/session-quiz-question';
import { CommonError } from '@core/error-handler/common-error';
import { CommonHttpErrorService } from '@core/error-handler/common-http-error.service';
import { Observable, catchError } from 'rxjs';

@Injectable({
  providedIn: 'root'
})
export class EmployeeTrainingService {
  private httpClient = inject(HttpClient)
  private commonHttpErrorService = inject(CommonHttpErrorService);

  getEmployeeCourseLink(link: string): Observable<EmployeeCourse | CommonError> {
    const url = `EmployeeCourse/link/${link}`
    return this.httpClient
      .get<EmployeeCourse>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  addEmployeeCourseSessionQuizAnswer(courseSessionQuizAnswer: QuizResponse): Observable<EmployeeCourseSessionQuiz | CommonError> {
    const url = 'EmployeeCourseSessionQuizAnswer'
    return this.httpClient
      .post<EmployeeCourseSessionQuiz>(url, courseSessionQuizAnswer)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  addEmployeeCourseSessionActivity(courseSessionActivity: EmployeeCourseSessionActivity): Observable<void | CommonError> {
    const url = 'EmployeeCourseSessionActivity'
    return this.httpClient
      .post<void>(url, courseSessionActivity)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  getCourseSessionQuiz(id: string): Observable<SessionQuiz | CommonError> {
    const url = `CourseSessionQuiz/${id}`
    return this.httpClient
      .get<SessionQuiz>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  addEmployeeCourseSessionMedia(empooyeeCourseId: string, mediaId: string): Observable<void | CommonError> {
    const url = 'EmployeeCourseSessionMedia';
    return this.httpClient
      .post<void>(url, { employeeCourseId: empooyeeCourseId, mediaId: mediaId })
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  getResumeCourseSessionMedia(empooyeeCourseLink: string): Observable<ResumeCourseSessionMedia | CommonError> {
    const url = `EmployeeCourseSession/session/media/${empooyeeCourseLink}`;
    return this.httpClient
      .get<ResumeCourseSessionMedia>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  getCourseSessionMedia(link: string, sessionId: string): Observable<CourseSessionMedia[] | CommonError> {
    const url = `EmployeeCourseSession/status/${link}/${sessionId}`;
    return this.httpClient
      .get<CourseSessionMedia[]>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  getEmployeeCourseCertificate(id: string): Observable<HttpEvent<Blob> | CommonError> {
    const url = `EmployeeCourse/${id}/download-certificate`;
    return this.httpClient
      .get(url, {
        reportProgress: true,
        observe: 'events',
        responseType: 'blob',
      })
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }
}
