import { NgClass } from '@angular/common';
import { Component, inject, OnInit } from '@angular/core';
import { TranslateModule } from '@ngx-translate/core';
import { QuizFormResponseQuestion, QuizFormResponse, QuizQuestion, QuizResponse } from '@core/domain-classes/session-quiz-question';
import { FormArray, FormBuilder, FormGroup, ReactiveFormsModule } from '@angular/forms';
import { SessionQuiz } from '@core/domain-classes/session-quiz';
import { ActivatedRoute, Router, RouterLink } from '@angular/router';
import { EmployeeTrainingService } from '../employee-training.service';

import { ToastrService } from '@core/services/toastr-service';
import { CommonDialogService } from '@core/common-dialog/common-dialog.service';
import { EmployeeCourseSessionQuiz } from '@core/domain-classes/employee-course-session-quiz';
import { BaseComponent } from '../../../base.component';
import { MatButtonModule } from '@angular/material/button';
import { MatIconModule } from '@angular/material/icon';
@Component({
  selector: 'app-employee-session-quiz',
  imports: [
    ReactiveFormsModule,
    TranslateModule,
    MatButtonModule,
    RouterLink,
    NgClass,
    MatIconModule
  ],
  templateUrl: './session-quiz.component.html',
  styleUrl: './session-quiz.component.scss'
})

export class SessionQuizComponent extends BaseComponent implements OnInit {
  fb = inject(FormBuilder);
  trainingQuizFrom!: FormGroup;
  sessionQuiz: SessionQuiz;
  route = inject(ActivatedRoute);
  employeeCourseService = inject(EmployeeTrainingService);
  toastr = inject(ToastrService);
  sessionName: string | undefined;
  commandDialogService = inject(CommonDialogService);
  router = inject(Router);
  answeredCount: number = 0;

  get questionsArray() {
    return this.trainingQuizFrom.get('employeeCourseSessionQuizQuestions') as FormArray;
  }

  getSelectedOptionsArray(i: number): FormArray {
    return this.questionsArray.at(i).get('selectedOptions') as FormArray;
  }

  ngOnInit(): void {
    this.creattrainingQuizForm();
    this.calculateAnsweredCount();

    this.sub$.sink = this.route.params.subscribe((params) => {
      const link = params['link'];
      const courseSessionId = params['courseSessionId'];
      this.trainingQuizFrom.patchValue({ link, courseSessionId });
    });

    this.route.data.subscribe((response) => {
      const sessionQuiz: SessionQuiz = response['sessionQuiz'];
      if (sessionQuiz) {
        this.sessionQuiz = sessionQuiz;
        this.sessionName = sessionQuiz.courseSessionName;
        sessionQuiz.quizQuestions.forEach((question: QuizQuestion) => {
          const optionsFormArray = this.fb.array(
            question.quizQuestionOptions.map(option =>
              this.fb.group({
                optionId: [option.id],
                optionText: [option.optionText],
                isSelected: [false]
              })
            )
          );
          this.questionsArray.push(
            this.fb.group({
              questionId: [question.id],
              selectedOptions: optionsFormArray
            })
          );
        });
      }
    });
  }

  calculateAnsweredCount(): void {
    this.answeredCount = this.trainingQuizFrom.value.employeeCourseSessionQuizQuestions
      .filter((q: any) => q.selectedOptions.some((opt: any) => opt.isSelected)).length;
  }

  onOptionChange(): void {
    this.calculateAnsweredCount();
  }

  creattrainingQuizForm() {
    this.trainingQuizFrom = this.fb.group({
      id: [''],
      courseSessionId: [''],
      link: [''],
      employeeCourseSessionQuizQuestions: this.fb.array([])
    })
  }

  saveEmployeeCourseSeesionQuiz() {
    const quizData: QuizFormResponse = this.trainingQuizFrom.getRawValue();
    const selectedQuizQuestion: QuizFormResponseQuestion[] = quizData.employeeCourseSessionQuizQuestions.filter((as) => as.selectedOptions.some((o) => o.isSelected)).map((d) => {
      return {
        questionId: d.questionId,
        selectedOptions: d.selectedOptions.filter((s) => s.isSelected)
      }
    });

    quizData.employeeCourseSessionQuizQuestions = selectedQuizQuestion;

    const request: QuizResponse = {
      courseSessionId: quizData.courseSessionId,
      link: quizData.link,
      employeeCourseSessionQuizQuestions: quizData.employeeCourseSessionQuizQuestions.map(d => {
        return {
          questionId: d.questionId,
          selectedOptionIds: d.selectedOptions.map(o => o.optionId)
        }
      })
    }

    this.employeeCourseService.addEmployeeCourseSessionQuizAnswer(request).subscribe({
      next: (response) => {
        const CourseSession = response as EmployeeCourseSessionQuiz;
        if (CourseSession) {
          this.toastr.success(this.translationService.getValue('QUIZ_SAVE_SUCCESSFULLY'));
          this.router.navigate([`course/${request.link}`]);
        }
      },
      error: (err) =>
        this.toastr.error(err.error)
    });
  }
}
