import { Component, inject, OnInit } from '@angular/core';
import { ActivatedRoute, Data, RouterLink } from '@angular/router';
import { EmployeeCourse } from '@core/domain-classes/employee-course-session';
import { MediaPreview } from '@core/domain-classes/media-previw-data';
import { BasePreviewMediaComponent } from '@shared/base-preview-media/base-preview-media.component';
import { BaseComponent } from '../../../base.component';
import { CourseSessionMedia } from '@core/domain-classes/course-session-media';
import { EmployeeTrainingService } from '../employee-training.service';
import { MatIconModule } from '@angular/material/icon';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { TranslateModule } from '@ngx-translate/core';
import { MatCardModule } from '@angular/material/card';
import { MatButtonModule } from '@angular/material/button';
import { EmployeeCourseStatus } from '../employee-course-status.enum';
import { QmsModuleEnum } from '@core/domain-classes/qms-module-enum';

@Component({
  selector: 'app-training-course',
  imports: [
    BasePreviewMediaComponent,
    RouterLink,
    MatIconModule,
    PageHelpTextComponent,
    TranslateModule,
    MatCardModule,
    MatButtonModule
  ],
  templateUrl: './training-course.component.html',
  styleUrl: './training-course.component.scss'
})
export class TrainingCourseComponent extends BaseComponent implements OnInit {
  route = inject(ActivatedRoute);
  link = '';
  employeeCourseSessionData: EmployeeCourse;
  sessionMedia: Partial<MediaPreview>;
  employeeTrainingService = inject(EmployeeTrainingService);
  mediaList: CourseSessionMedia[] = [];
  documentId: string;
  expandedIndex: number | null = null;
  courseStatus = EmployeeCourseStatus;

  ngOnInit(): void {
    this.sub$.sink = this.route.data.subscribe((response: Data) => {
      if (response['CourseSessionDetails']) {
        this.employeeCourseSessionData = response['CourseSessionDetails'];
      }
    });

    this.sub$.sink = this.route.queryParams.subscribe(params => {
      this.documentId = params['documentId'];
    });

    this.sub$.sink = this.route.params.subscribe(params => {
      this.link = params['link'];
      const sessionId = params['id'];
      if (sessionId) {
        const index = this.employeeCourseSessionData?.courseSessions?.findIndex(cs => cs.id === sessionId) ?? 0;
        this.onAccordionToggle(index, sessionId);
      } else {
        this.mediaList = [];
      }
    });
  }

  getCourseSessionMedia(sessionId: string) {
    this.employeeTrainingService.getCourseSessionMedia(this.link, sessionId).subscribe({
      next: (resp) => {
        const sessionMedia = resp as CourseSessionMedia[];
        if (sessionMedia) {
          this.mediaList = [...sessionMedia];
          if (this.documentId != null) {
            const media = sessionMedia.find(m => m.id === this.documentId);
            if (media) {
              this.onMeidaView(media);
            }
          } else {
            this.onMeidaView(this.mediaList[0]);
          }
        }
      }
    })
  }

  onAccordionToggle(index: number, sessionId: string) {
    this.expandedIndex = this.expandedIndex === index ? null : index;

    if (this.expandedIndex !== null && sessionId) {
      this.getCourseSessionMedia(sessionId);
    }
  }

  onMeidaView(media: CourseSessionMedia) {
    this.employeeTrainingService.addEmployeeCourseSessionMedia(this.employeeCourseSessionData.id, media.id).subscribe((r) => { });
    this.sessionMedia = {
      id: media.id,
      name: media.name,
      mediaType: media.mediaType,
      url: media.url,
      extension: media.extension,
      isAllChunkUploaded: media.isAllChunkUploaded,
      isChunked: media.isChunked,
      isTracking: true,
      employeeCourseSessionLink: this.link,
      isCourseSessionMedia: true,
      moduleNo: QmsModuleEnum.Training
    };
    if (media != null) {
      this.documentId = media.id;
    }
  }

  getFileIcon(fileType: string | undefined): string {
    if (!fileType) {
      return 'insert_drive_file';
    }
    if (fileType.startsWith('image/')) {
      return 'image';
    } else if (fileType.includes('pdf')) {
      return 'picture_as_pdf';
    } else if (fileType.includes('spreadsheet') || fileType.includes('excel')) {
      return 'table_chart';
    } else if (fileType.includes('document') || fileType.includes('word')) {
      return 'description';
    } else {
      return 'insert_drive_file';
    }
  }
}
