import { Component, ElementRef, inject, OnInit, ViewChild } from '@angular/core';
import { CourseSessionMedia, ResumeCourseSessionMedia } from '@core/domain-classes/course-session-media';
import { ActivatedRoute, Data, Router, RouterLink } from '@angular/router';
import { SessionMediaService } from '../../course-session/session-media/session-meida.service';
import { BaseComponent } from '../../../base.component';
import { EmployeeCourse } from '@core/domain-classes/employee-course-session';
import { EmployeeTrainingService } from '../employee-training.service';
import { MatIconModule } from '@angular/material/icon';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { TranslateModule } from '@ngx-translate/core';
import { FormsModule } from '@angular/forms';
import { TextEditorComponent } from '@shared/text-editor/text-editor.component';
import { EmployeeCourseStatus } from '../employee-course-status.enum';
import { EmployeeCourseStatusPipe } from '../../pipes/employee-course-status.pipe';
import { MatCardModule } from '@angular/material/card';
import { MatButtonModule } from '@angular/material/button';

import { ToastrService } from '@core/services/toastr-service';
import { HttpEvent, HttpEventType } from '@angular/common/http';
import { CommonError } from '@core/error-handler/common-error';

@Component({
  selector: 'app-training',
  imports: [
    RouterLink,
    MatIconModule,
    PageHelpTextComponent,
    TranslateModule,
    TextEditorComponent,
    FormsModule,
    EmployeeCourseStatusPipe,
    MatCardModule,
    MatButtonModule
  ],
  templateUrl: './training.component.html',
  styleUrl: './training.component.scss',
})

export class TrainingComponent extends BaseComponent implements OnInit {
  employeeCourseSessionData: EmployeeCourse;
  route = inject(ActivatedRoute);
  link = '';
  sessionMediaService = inject(SessionMediaService);
  mediaList: CourseSessionMedia[] = [];
  router = inject(Router);
  employeeTrainingService = inject(EmployeeTrainingService);
  toaster = inject(ToastrService);
  courseStatus = EmployeeCourseStatus;
  expandedIndex: number | null = null;

  @ViewChild('certificate', { static: false }) certificateEl!: ElementRef;


  ngOnInit(): void {
    this.sub$.sink = this.route.params.subscribe(params => {
      this.link = params['link'];
    });

    this.sub$.sink = this.route.data.subscribe((response: Data) => {
      if (response['CourseSessionDetails']) {
        this.employeeCourseSessionData = response['CourseSessionDetails'];
      }
    });
  }

  toggleSession(index: number, sessionId: string) {
    this.expandedIndex = this.expandedIndex === index ? null : index;

    if (this.expandedIndex !== null && sessionId) {
      this.getCourseSession(sessionId);
    }
  }

  getCourseSession(sessionId: string) {
    this.employeeTrainingService.getCourseSessionMedia(this.link, sessionId).subscribe({
      next: (resp) => {
        const sessionMedia = resp as CourseSessionMedia[];
        if (sessionMedia) {
          this.mediaList = [...sessionMedia];
        }
      },
      error: (err) => {
        console.error('Error fetching course session media:', err);
        this.mediaList = [];
      }
    });
  }

  getResumeCourseSessionMedia(link: string): void {
    this.employeeTrainingService.getResumeCourseSessionMedia(link).subscribe({
      next: (resp) => {
        const resumeMedia = resp as ResumeCourseSessionMedia;
        if (resumeMedia) {
          this.router.navigate(
            ['/training/', link, resumeMedia.courseSessionId],
            { queryParams: { documentId: resumeMedia.courseSessionMediaId } }
          );
        }
      },
    });
  }

  getFileIcon(fileType: string | undefined): string {
    if (!fileType) {
      return 'insert_drive_file';
    }
    if (fileType.startsWith('image/')) {
      return 'image';
    } else if (fileType.includes('pdf')) {
      return 'picture_as_pdf';
    } else if (fileType.includes('spreadsheet') || fileType.includes('excel')) {
      return 'table_chart';
    } else if (fileType.includes('document') || fileType.includes('word')) {
      return 'description';
    } else {
      return 'insert_drive_file';
    }
  }

  downloadCertificate(courseId: string): void {
    this.sub$.sink = this.employeeTrainingService.getEmployeeCourseCertificate(courseId).subscribe({
      next: (response: HttpEvent<Blob> | CommonError) => {
        const event = response as HttpEvent<Blob>;
        if (event && event.type === HttpEventType.Response) {
          const blobParth = event?.body as BlobPart;
          const blog = new Blob([blobParth], { type: event?.body?.type });
          this.downloadFile(blog);
        }
      },
      error: (err) => {
        this.toaster.error('Failed to download certificate.');
      }
    });
  }

  private downloadFile(blog: Blob) {
    const a = document.createElement('a');
    a.setAttribute('style', 'display:none;');
    document.body.appendChild(a);
    a.download = 'Employee_Certificate.pdf';
    a.href = URL.createObjectURL(blog);
    a.target = '_blank';
    a.click();
    document.body.removeChild(a);
  }

  startCourse() {
    if (
      this.employeeCourseSessionData &&
      this.employeeCourseSessionData.courseSessions &&
      this.employeeCourseSessionData.courseSessions.length > 0
    ) {
      const firstSessionId = this.employeeCourseSessionData.courseSessions[0].id;
      const queryParams =
        this.mediaList && this.mediaList.length > 0
          ? { documentId: this.mediaList[0].id }
          : null;

      this.router.navigate(
        ['/training/', this.link, firstSessionId],
        { queryParams }
      );
    } else {
      this.toaster.error('No course sessions available to start the course.');
    }
  }
}
