import { Component, inject, Inject, OnInit } from '@angular/core';
import { MAT_DIALOG_DATA, MatDialogModule, MatDialogRef } from '@angular/material/dialog';
import { BaseComponent } from '../../../base.component';
import { CourseService } from '../../course/course.service';
import { TranslateModule } from '@ngx-translate/core';
import { MatSelectModule } from '@angular/material/select';
import { FormControl, ReactiveFormsModule } from '@angular/forms';
import { debounceTime, distinctUntilChanged, switchMap } from 'rxjs';
import { IdName } from '@core/domain-classes/id-name';
import { Employee } from '@core/domain-classes/employee';
import { MatButtonModule } from '@angular/material/button';

import { ToastrService } from '@core/services/toastr-service';
import { AssignCourseCompanyRequest } from '@core/domain-classes/assign-course-company-request';
import { MatIconModule } from '@angular/material/icon';

@Component({
  selector: 'app-employee-assign-course',
  imports: [
    MatDialogModule,
    TranslateModule,
    MatSelectModule,
    ReactiveFormsModule,
    MatButtonModule,
    MatIconModule
  ],
  templateUrl: './employee-assign-course.component.html',
  styleUrl: './employee-assign-course.component.css'
})
export class EmployeeAssignCourseComponent extends BaseComponent implements OnInit {
  courseList: IdName[] = [];
  courseService = inject(CourseService);
  searchCourseControl = new FormControl();
  courseId = new FormControl();
  toastr = inject(ToastrService);

  constructor(
    public dialogRef: MatDialogRef<EmployeeAssignCourseComponent>,
    @Inject(MAT_DIALOG_DATA) public employee: Employee,
  ) {
    super();
  }

  ngOnInit(): void {
    this.courseNameChangeValue();
    this.searchCourseControl.setValue('');
  }

  courseNameChangeValue() {
    this.sub$.sink = this.searchCourseControl.valueChanges.pipe(
      debounceTime(500),
      distinctUntilChanged(),
      switchMap((c) => {
        return this.courseService.getCoursesForDropdown(c ?? '');
      })
    ).subscribe(
      (resp) => {
        this.courseList = resp as IdName[];
      });
  }

  onSubmit() {
    const courseId: string = this.courseId.value;
    if (!courseId || !this.employee.id) {
      this.toastr.error(this.translationService.getValue('PLEASE_SELECTE_A_COURSE'));
      return;
    }

    const assignCourse: AssignCourseCompanyRequest = {
      employeeIds: [this.employee.id],
      courseId: courseId
    }

    if (courseId) {
      this.courseService.assignCourseToEmployees(assignCourse).subscribe({
        next: () => {
          this.toastr.success(this.translationService.getValue('COURSE_ASSIGNED_SUCCESSFULLY'));
          this.dialogRef.close(true);
        },
        error: (err) => {
          console.error('Error assigning course:', err);
        }
      });
    }
  }

  onClick(): void {
    this.dialogRef.close();
  }
}
