import { Component, inject, OnInit } from '@angular/core';
import { MatIconModule } from '@angular/material/icon';
import { MatTableModule } from '@angular/material/table';
import { TranslateModule } from '@ngx-translate/core';
import { trigger, state, style, transition, animate } from '@angular/animations';
import { EmployeeService } from '../employee.service';
import { ActivatedRoute, RouterLink } from '@angular/router';
import { Employee } from '@core/domain-classes/employee';
import { AssignedCourse, CourseSession } from '@core/domain-classes/employee-assigned-courses';
import { MatDialog } from '@angular/material/dialog';
import { BaseComponent } from '../../../base.component';
import { SessionService } from '../../course-session/course-session.service';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { MatButtonModule } from '@angular/material/button';
import { UTCToLocalTime } from '@shared/pipes/utc-to-localtime.pipe';
import { MatCardModule } from '@angular/material/card';
import { SecurityService } from '@core/security/security.service';
import { User } from '@core/domain-classes/user';
import { NgClass } from '@angular/common';

@Component({
  selector: 'app-employee-details',
  imports: [
    TranslateModule,
    MatTableModule,
    MatIconModule,
    MatButtonModule,
    RouterLink,
    PageHelpTextComponent,
    UTCToLocalTime,
    MatCardModule,
    MatIconModule,
    NgClass
  ],
  templateUrl: './employee-details.component.html',
  styleUrl: './employee-details.component.scss',
  animations: [
    trigger('detailExpand', [
      state('collapsed', style({ height: '0px', minHeight: '0' })),
      state('expanded', style({ height: '*' })),
      transition('expanded <=> collapsed', animate('225ms cubic-bezier(0.4, 0.0, 0.2, 1)')),
    ]),
  ]
})
export class EmployeeDetailsComponent extends BaseComponent implements OnInit {
  dataSourse: AssignedCourse[] = [];
  courseSession: CourseSession[] = [];
  displayedColumns = ['action', 'title', 'status'];
  subActionColumnToDisplay = ['session', 'completedDate', 'result', 'scorePercentage'];
  expandedElement!: AssignedCourse | null;
  employeeService = inject(EmployeeService);
  route = inject(ActivatedRoute);
  employee!: User;
  courseSessionService = inject(SessionService);
  isTimeline = false;
  dialog = inject(MatDialog);
  securityService = inject(SecurityService);

  ngOnInit(): void {
    this.sub$.sink = this.route.data.subscribe((data) => {
      this.employee = data['employee'] as User;
      if (this.employee) {
        this.getAssignesCourses(this.employee.id ?? '');
      }
    });
  }

  getAssignesCourses(employeeId: string) {
    this.sub$.sink = this.employeeService.getAssignedCourseByEmployeeId(employeeId).subscribe({
      next: (response) => {
        const course = response as AssignedCourse[]
        if (course) {
          this.dataSourse = course
        }
      }
    })
  }

  getCourseSession(course: AssignedCourse) {
    if (course) {
      this.expandedElement = this.expandedElement === course ? null : course
      if (!this.expandedElement) {
        return;
      }
      this.sub$.sink = this.employeeService.getSessionByCourseId(course.courseId, course.employeeId).subscribe({
        next: (response) => {
          const assignedCourse = response as CourseSession[]
          if (assignedCourse) {
            this.courseSession = assignedCourse
          }
        }
      })
    }
  }

  // Statistics calculation methods
  getTotalAssignedCourses(): number {
    return this.dataSourse.length;
  }

  getCompletedCourses(): number {
    return this.dataSourse.filter(course => course.status === 2).length;
  }

  getInProgressCourses(): number {
    return this.dataSourse.filter(course => course.status === 1).length;
  }

  getPendingCourses(): number {
    return this.dataSourse.filter(course => course.status === 0).length;
  }

  isOddDataRow(index: number): boolean {
    // index = the index in dataSource, not in DOM
    return index % 2 !== 0;
  }

  getDataIndex(row: any) {
    return this.dataSourse.indexOf(row);
  }
}
