import { AfterViewInit, Component, inject, OnInit, ViewChild } from '@angular/core';
import { MatDialog } from '@angular/material/dialog';
import { MatPaginator } from '@angular/material/paginator';
import { MatSort, MatSortModule } from '@angular/material/sort';
import { CommonDialogService } from '@core/common-dialog/common-dialog.service';
import { Employee } from '@core/domain-classes/employee';
import { Subject, debounceTime, distinctUntilChanged, merge, tap } from 'rxjs';
import { ManageEmployeeComponent } from '../manage-employee/manage-employee.component';
import { BaseComponent } from '../../../base.component';
import { PageSizeOption } from '@core/utils/global-config';
import { EmployeeStore } from '../store/employee.store';
import { TranslateModule } from '@ngx-translate/core';
import { MatTableModule } from '@angular/material/table';
import { RouterLink } from '@angular/router';
import { FormsModule } from '@angular/forms';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { EmployeeAssignCourseComponent } from '../employee-assign-course/employee-assign-course.component';
import { MatMenuModule } from '@angular/material/menu';
import { MatIconModule } from '@angular/material/icon';
import { MatCardModule } from '@angular/material/card';
import { MatButtonModule } from '@angular/material/button';
import { NgClass } from '@angular/common';

@Component({
  selector: 'app-employee-list',
  imports: [
    TranslateModule,
    MatTableModule,
    MatPaginator,
    MatSortModule,
    RouterLink,
    FormsModule,
    PageHelpTextComponent,
    HasClaimDirective,
    MatMenuModule,
    MatIconModule,
    MatCardModule,
    MatButtonModule,
    NgClass
  ],
  templateUrl: './employee-list.component.html',
  styleUrl: './employee-list.component.scss'
})
export class EmployeeListComponent extends BaseComponent implements OnInit, AfterViewInit {
  private dialog = inject(MatDialog);
  pageOption = PageSizeOption;
  displayedColumns: string[] = [
    'action',
    'employeeNo',
    'firstName',
    'lastName',
    'email'
  ];
  footerToDisplayed = ['footer'];
  employeeStore = inject(EmployeeStore);
  @ViewChild(MatPaginator) paginator!: MatPaginator;
  @ViewChild(MatSort) sort!: MatSort;
  commandDialogService = inject(CommonDialogService);
  employeeSearchParameter = { ...this.employeeStore.filterData() };
  filterParameter$: Subject<string> = new Subject<string>();
  _filterEmployeeNo = this.employeeSearchParameter.employeeNo;
  _filterFirstName = this.employeeSearchParameter.firstName;
  _filterLastName = this.employeeSearchParameter.lastName;
  _filterEmail = this.employeeSearchParameter.email;

  get employeeNoFilter() {
    return this._filterEmployeeNo;
  }

  set employeeNoFilter(character: string) {
    if (this._filterEmployeeNo != character) {
      this._filterEmployeeNo = character;
      const filterEmployeeNo = `filterEmployeeNo#${character}`;
      this.filterParameter$.next(filterEmployeeNo);
    }
  }
  get firstNameFilter() {
    return this._filterFirstName;
  }

  set firstNameFilter(character: string) {
    if (this._filterFirstName != character) {
      this._filterFirstName = character;
      const filtername = `filterFirstName#${character}`;
      this.filterParameter$.next(filtername);
    }
  }

  get lastNameFilter() {
    return this._filterLastName;
  }

  set lastNameFilter(character: string) {
    if (this._filterLastName != character) {
      this._filterLastName = character;
      const filterLastname = `filterLastName#${character}`;
      this.filterParameter$.next(filterLastname);
    }
  }

  get emailFilter() {
    return this._filterEmail;
  }

  set emailFilter(character: string) {
    if (this._filterEmail != character) {
      this._filterEmail = character;
      const filterEmail = `filterEmail#${character}`;
      this.filterParameter$.next(filterEmail);
    }
  }

  ngOnInit(): void {
    this.sub$.sink = this.filterParameter$
      .pipe(
        debounceTime(1000),
        distinctUntilChanged()
      ).subscribe((c: string) => {
        this.employeeSearchParameter.skip = 0;
        this.paginator.pageIndex = 0;
        const filterArray: Array<string> = c.split('#')
        if (filterArray[0] === 'filterEmployeeNo') {
          this.employeeSearchParameter.employeeNo = filterArray[1]
        }
        else if (filterArray[0] === 'filterFirstName') {
          this.employeeSearchParameter.firstName = filterArray[1]
        }
        else if (filterArray[0] === 'filterLastName') {
          this.employeeSearchParameter.lastName = filterArray[1]
        }
        else if (filterArray[0] === 'filterEmail') {
          this.employeeSearchParameter.email = filterArray[1]
        }
        this.employeeStore.loadByQuery(this.employeeSearchParameter);
      });
  }

  addemployee() {
    this.dialog.open(ManageEmployeeComponent, {
      width: '500px',
    })
  }

  updateEmployee(employee: Employee) {
    this.dialog.open(ManageEmployeeComponent, {
      width: '500px',
      data: Object.assign({}, employee)
    })
  }


  employeeAssignCourse(employee: Employee) {
    this.dialog.open(EmployeeAssignCourseComponent, {
      width: '550px',
      data: Object.assign({}, employee)
    })
  }

  deleteEmployee(employeeId: string) {
    this.commandDialogService.deleteConfirmtionDialog(this.translationService.getValue("ARE_YOU_SURE_YOU_WANT_TO_DELETE_THIS_EMPLOYEE")).subscribe({
      next: (result: boolean) => {
        if (result) {
          this.employeeStore.deleteEmployee(employeeId);
        }
      }
    });
  }

  ngAfterViewInit() {
    this.sub$.sink = this.sort.sortChange.subscribe(() => {
      this.paginator.pageIndex = 0
    })
    this.sub$.sink = merge(this.sort.sortChange, this.paginator.page)
      .pipe(
        tap(() => {
          this.employeeSearchParameter.skip =
            this.paginator.pageIndex * this.paginator.pageSize;
          this.employeeSearchParameter.pageSize = this.paginator.pageSize;
          this.employeeSearchParameter.orderBy =
            this.sort.active + ' ' + this.sort.direction;
          this.employeeStore.loadByQuery(this.employeeSearchParameter)
        })
      ).subscribe()
  }

  refresh() {
    this.employeeStore.loadByQuery(this.employeeSearchParameter);
  }

  isOddDataRow(index: number): boolean {
    // index = the index in dataSource, not in DOM
    return index % 2 !== 0;
  }

  getDataIndex(row: any) {
    return this.employeeStore.employeeList().indexOf(row);
  }
}
