import { HttpParams } from '@angular/common/http';
import { inject, Injectable } from '@angular/core';
import { Employee } from '@core/domain-classes/employee';
import { EmployeeResource } from '@core/domain-classes/employee-resource';
import { CommonError } from '@core/error-handler/common-error';
import { CommonHttpErrorService } from '@core/error-handler/common-http-error.service';
import { HttpClient, HttpResponse } from '@angular/common/http';
import { Observable, catchError } from 'rxjs';
import { EmployeeImportResponse } from '@core/domain-classes/employee-import.response';
import { AssignedCourse, CourseSession } from '@core/domain-classes/employee-assigned-courses';
import { EmployeeCourseSession } from '@core/domain-classes/employee-course-session';
import { EmployeeCourseSessionActivityResponse } from '@core/domain-classes/employee-course-session-activity';

@Injectable({
  providedIn: 'root'
})
export class EmployeeService {
  private httpClient = inject(HttpClient)
  private commonHttpErrorService = inject(CommonHttpErrorService)
  constructor() { }

  getEmployees(
    resource: EmployeeResource
  ): Observable<HttpResponse<Employee[]>> {
    const url = 'Employee';
    const customParams = new HttpParams()
      .set('OrderBy', resource.orderBy)
      .set('PageSize', resource.pageSize)
      .set('Skip', resource.skip)
      .set('FirstName', resource.firstName.toString())
      .set('LastName', resource.lastName.toString())
      .set('Email', resource.email.toString())
      .set('EmployeeNo', resource.employeeNo)
    return this.httpClient
      .get<Employee[]>(url, {
        params: customParams,
        observe: 'response',
      })
  }

  addEmployee(employee: Employee): Observable<Employee | CommonError> {
    const url = 'Employee';
    return this.httpClient
      .post<Employee>(url, employee)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  updateEmployee(employee: Employee): Observable<Employee | CommonError> {
    const url = `Employee/${employee.id}`
    return this.httpClient
      .put<Employee>(url, employee)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  deleteEmployee(employeeId: string): Observable<void | CommonError> {
    const url = `Employee/${employeeId}`
    return this.httpClient
      .delete<void>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  importEmployee(employees: Employee[]): Observable<EmployeeImportResponse[] | void | CommonError> {
    const url = 'Employee/Import';
    return this.httpClient
      .post<EmployeeImportResponse[]>(url, {
        employees: employees,
      })
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  getAssignedCourseByEmployeeId(employeeId: string): Observable<AssignedCourse[] | CommonError> {
    const url = `EmployeeCourse/${employeeId}`;
    return this.httpClient
      .get<AssignedCourse[]>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  getSessionByCourseId(courseId: string, employeeId: string): Observable<CourseSession[] | CommonError> {
    const url = `EmployeeCourseSession/${courseId}/sessions/${employeeId}`;
    return this.httpClient
      .get<CourseSession[]>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  getEmployeeId(employeeId: string): Observable<Employee | CommonError> {
    const url = `Employee/${employeeId}`
    return this.httpClient
      .get<Employee>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  getEmployeeCourseSessionDetails(employeeCourseSessionId: string): Observable<EmployeeCourseSession | CommonError> {
    const url = `EmployeeCourseSession/${employeeCourseSessionId}/detail`;
    return this.httpClient
      .get<EmployeeCourseSession>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }

  getEmployeeCourseSessionActivityes(employeeCourseSessionId: string): Observable<EmployeeCourseSessionActivityResponse[] | CommonError> {
    const url = `EmployeeCourseSessionActivity/${employeeCourseSessionId}`;
    return this.httpClient
      .get<EmployeeCourseSessionActivityResponse[]>(url)
      .pipe(catchError(this.commonHttpErrorService.handleError));
  }
}
