import { Component, inject } from '@angular/core';
import { FormGroup, FormBuilder, Validators, FormArray, ReactiveFormsModule } from '@angular/forms';
import { Router } from '@angular/router';
import { Employee } from '@core/domain-classes/employee';
import { EmployeeImportResponse } from '@core/domain-classes/employee-import.response';
import { TranslationService } from '@core/services/translation.service';

import { ToastrService } from '@core/services/toastr-service';
import { EmployeeService } from '../employee.service';
import { EmployeeStore } from '../store/employee.store';
import { TranslateModule } from '@ngx-translate/core';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { MatButtonModule } from '@angular/material/button';
import { MatIconModule } from '@angular/material/icon';
import { MatCardModule } from '@angular/material/card';

@Component({
  selector: 'app-import-employee',
  imports: [
    TranslateModule,
    ReactiveFormsModule,
    PageHelpTextComponent,
    MatButtonModule,
    MatIconModule,
    MatCardModule
  ],
  templateUrl: './import-employee.component.html',
  styleUrl: './import-employee.component.scss'
})
export class ImportEmployeeComponent {
  employees: Employee[] = []
  toasrt = inject(ToastrService);
  employeeForm: FormGroup;
  fb = inject(FormBuilder);
  employeeService = inject(EmployeeService);
  router = inject(Router);
  translationService = inject(TranslationService);
  employeeStore = inject(EmployeeStore);
  employeeSearchParameter = { ...this.employeeStore.filterData() };

  constructor() {
    this.employeeForm = this.fb.group({
      employees: this.fb.array([]),
    });
  }

  get employeeFormArray() {
    return this.employeeForm.get('employees') as FormArray;
  }

  removeEmployee(index: number) {
    this.employeeFormArray.removeAt(index);
  }

  importEmployee() {
    if (!this.employeeForm.valid) {
      this.employeeForm.markAllAsTouched();
      return;
    } else {
      const employees = this.employeeFormArray.value;
      this.employeeService.importEmployee(employees).subscribe({
        next: (response) => {
          const employeeImportResponse = response as EmployeeImportResponse[];
          if (employeeImportResponse && employeeImportResponse?.length > 0) {
            this.employeeFormArray.clear();
            employeeImportResponse.forEach((employee) => {
              this.employeeFormArray.push(this.fb.group({
                employeeNo: [employee.employeeNo],
                firstName: [employee.firstName, [Validators.required]],
                lastName: [employee.lastName, [Validators.required]],
                email: [employee.email, [Validators.required, Validators.email]],
                error: [employee.error]
              }));
            });
          } else {
            this.toasrt.success(this.translationService.getValue('EMPLOYEES_IMPORTED_SUCCESSFULLY'));
            this.router.navigate(['employee/list']);
            this.employeeStore.loadByQuery(this.employeeSearchParameter);
          }
        }
      });
    }
  }

  onFileUpload(event: Event) {
    const input = event.target as HTMLInputElement;
    const file = input?.files?.[0];
    if (file) {
      if (file.type !== 'text/csv') {
        this.toasrt.error('Please upload a valid CSV file.');
        return;
      }
      const reader = new FileReader();
      reader.onload = (e) => {
        const content = e.target?.result as string;
        const lines = content.split('\n');
        const parsedData = lines.slice(1).map(line => {
          const [employeeNo, firstName, lastName, email] = line.trim().split(',').map(value => value.trim());
          return { employeeNo, firstName, lastName, email };
        });
        this.employees = parsedData.filter(employee => employee.employeeNo || employee.firstName || employee.lastName || employee.email);
        this.employeeFormArray.clear();
        this.employees.forEach(employee => {
          this.employeeFormArray.push(this.fb.group({
            employeeNo: [employee.employeeNo],
            firstName: [employee.firstName, [Validators.required]],
            lastName: [employee.lastName, [Validators.required]],
            email: [employee.email, [Validators.required, Validators.email]],
            error: ['']
          }));
        });
      };
      reader.readAsText(file);
    }
  }
}
