import { Component, inject, Inject, OnInit } from '@angular/core';
import { toObservable } from '@angular/core/rxjs-interop';
import { FormGroup, FormBuilder, Validators, ReactiveFormsModule } from '@angular/forms';
import { MatDialogRef, MAT_DIALOG_DATA, MatDialogModule } from '@angular/material/dialog';
import { ActivatedRoute } from '@angular/router';
import { Employee } from '@core/domain-classes/employee';
import { EmployeeStore } from '../store/employee.store';
import { BaseComponent } from '../../../base.component';
import { TranslateModule } from '@ngx-translate/core';
import { MatButtonModule } from '@angular/material/button';
import { MatIconModule } from '@angular/material/icon';

@Component({
  selector: 'app-manage-employee',
  imports: [
    TranslateModule,
    MatDialogModule,
    ReactiveFormsModule,
    MatButtonModule,
    MatIconModule
  ],
  templateUrl: './manage-employee.component.html',
  styleUrl: './manage-employee.component.scss'
})
export class ManageEmployeeComponent extends BaseComponent implements OnInit {
  employeeForm!: FormGroup;
  dialogRef = inject(MatDialogRef<ManageEmployeeComponent>);
  fb = inject(FormBuilder);
  employeeStore = inject(EmployeeStore);
  route = inject(ActivatedRoute);
  isEdit = false;

  constructor(
    @Inject(MAT_DIALOG_DATA) public employeeDetail: Employee,
  ) {
    super();
    this.subscribeIsAddorUpdate();
  }

  ngOnInit(): void {
    this.creatEmployeeForm();
    if (this.employeeDetail) {
      this.isEdit = true
      this.employeeForm.patchValue(this.employeeDetail)
    }
  }

  creatEmployeeForm() {
    this.employeeForm = this.fb.group(
      {
        id: [''],
        employeeNo: [''],
        firstName: ['', [Validators.required]],
        lastName: ['', [Validators.required]],
        email: ['', [Validators.required, Validators.email]],
      }
    );
  }

  onSubmit() {
    if (!this.employeeForm.valid) {
      this.employeeForm.markAllAsTouched();
      return;
    }
    if (this.isEdit) {
      this.employeeStore.updateEmployee(this.employeeForm.getRawValue());
    } else {
      this.employeeStore.addEmployee(this.employeeForm.getRawValue());
    }
  }

  onNoClick(): void {
    this.dialogRef.close();
  }

  subscribeIsAddorUpdate() {
    toObservable(this.employeeStore.isAddorUpdate).subscribe((flag) => {
      if (flag) {
        this.onNoClick();
      }
      this.employeeStore.resetAddUpdate();
    })
  }
}
