import { inject } from "@angular/core"
import { patchState, signalStore, withHooks, withMethods, withState } from "@ngrx/signals"
import { ToastrService } from '@core/services/toastr-service';
import { EmployeeService } from "../employee.service"
import { Employee } from "@core/domain-classes/employee"
import { rxMethod } from "@ngrx/signals/rxjs-interop"
import { HttpResponse } from "@angular/common/http"
import { tapResponse } from "@ngrx/operators"
import { pipe, tap, switchMap } from "rxjs"
import { toObservable } from "@angular/core/rxjs-interop"
import { TranslationService } from "@core/services/translation.service"
import { EmployeeResource } from "@core/domain-classes/employee-resource"

type EmployeeState = {
  employeeList: Employee[],
  isAddorUpdate: boolean,
  loadList: boolean,
  filterData: EmployeeResource
}

const initialState: EmployeeState = {
  employeeList: [],
  isAddorUpdate: false,
  loadList: true,
  filterData: {
    employeeNo: '',
    name: '',
    email: '',
    firstName: '',
    lastName: '',
    skip: 0,
    pageSize: 10,
    orderBy: 'firstName asc',
    totalCount: 0,
  }
}

export const EmployeeStore = signalStore(
  { providedIn: 'root' },
  withState(initialState),
  withMethods((
    store,
    employeeService = inject(EmployeeService),
    toastr = inject(ToastrService),
    translationService = inject(TranslationService)
  ) => ({
    loadByQuery: rxMethod<EmployeeResource>(
      pipe(
        tap(() => patchState(store, { employeeList: [] })),
        switchMap((filter) => {
          return employeeService.getEmployees(filter).pipe(
            tapResponse({
              next: (httpResponse: HttpResponse<Employee[]>) => {
                if (httpResponse && httpResponse.body) {
                  patchState(store, { employeeList: [...httpResponse.body], loadList: false })
                }
                if (httpResponse.headers.get('x-pagination')) {
                  const pagination: EmployeeResource = JSON.parse(httpResponse.headers.get('x-pagination') ?? '');
                  patchState(store, { filterData: { ...filter, totalCount: pagination.totalCount } })
                }
              },
              error: () => {
              }
            })
          );
        })
      )
    ),
    addEmployee(employee: Employee) {
      employeeService.addEmployee(employee).subscribe({
        next: () => {
          patchState(store, { isAddorUpdate: true, loadList: true }),
            toastr.success(translationService.getValue('EMPLOYEE_SAVE_SUCCESSFULLY'))
        }
      })
    },
    updateEmployee(updateEmployee: Employee) {
      employeeService.updateEmployee(updateEmployee).subscribe({
        next: () => {
          patchState(store, {
            employeeList: store.employeeList().map((employee) =>
              updateEmployee.id === employee.id ? updateEmployee : employee),
            isAddorUpdate: true
          })
          toastr.success(translationService.getValue('EMPLOYEE_SAVE_SUCCESSFULLY'))
        }
      })
    },
    deleteEmployee(employeeId: string) {
      employeeService.deleteEmployee(employeeId).subscribe({
        next: () => {
          patchState(store, { loadList: true })
          toastr.success(translationService.getValue('EMPLOYEE_DELETE_SUCCESSFULLY'))
        }
      })
    },
    resetAddUpdate() {
      patchState(store, { isAddorUpdate: false })
    }
  })
  ),
  withHooks({
    onInit(store) {
      toObservable(store.loadList).subscribe((flag) => {
        if (flag) {
          store.loadByQuery(store.filterData());
        }
      })
    }
  })
)
