import { Component, inject, OnInit } from '@angular/core';
import { MatCardModule } from '@angular/material/card';
import { MatIconModule } from '@angular/material/icon';
import { MatButtonModule } from '@angular/material/button';
import { TranslateModule } from '@ngx-translate/core';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { CourseAssignment } from '@core/domain-classes/course';
import { MyCourses } from './my-courses';
import { NgClass, NgTemplateOutlet } from '@angular/common';
import { RouterLink } from '@angular/router';
import { LimitToPipe } from "../../shared/pipes/limit-to.pipe";
import { UTCToLocalTime } from "../../shared/pipes/utc-to-localtime.pipe";
import { forkJoin } from 'rxjs';
import { EmployeeTrainingService } from '../employee-training/employee-training.service';
import { HttpEvent, HttpEventType } from '@angular/common/http';
import { CommonError } from '@core/error-handler/common-error';

@Component({
  selector: 'app-my-courses-list',
  imports: [
    TranslateModule,
    MatCardModule,
    MatIconModule,
    MatButtonModule,
    PageHelpTextComponent,
    NgClass,
    NgTemplateOutlet,
    RouterLink,
    LimitToPipe,
    UTCToLocalTime
  ],
  templateUrl: './my-courses-list.html',
  styleUrl: './my-courses-list.scss'
})
export class MyCoursesList implements OnInit {
  myCourseServices = inject(MyCourses);
  employeeTrainingService = inject(EmployeeTrainingService);
  pendingCourses: CourseAssignment[] = [];
  inProgressCourses: CourseAssignment[] = [];
  completedCourses: CourseAssignment[] = [];

  ngOnInit(): void {
    this.getAllCourses();
  }

  getAllCourses(): void {
    forkJoin({
      pending: this.myCourseServices.getAllPendingCourses(),
      inProgress: this.myCourseServices.getAllInProgressCourses(),
      completed: this.myCourseServices.getAllCompletedCourses()
    }).subscribe(({ pending, inProgress, completed }) => {
      if (Array.isArray(pending)) {
        this.pendingCourses = pending;
      }
      if (Array.isArray(inProgress)) {
        this.inProgressCourses = inProgress;
      }
      if (Array.isArray(completed)) {
        this.completedCourses = completed;
      }
    });
  }

  downloadCertificate(courseId: string): void {
    this.employeeTrainingService.getEmployeeCourseCertificate(courseId).subscribe({
        next: (response: HttpEvent<Blob> | CommonError) => {
          const event = response as HttpEvent<Blob>;
          if (event && event.type === HttpEventType.Response) {
            const blobParth = event?.body as BlobPart;
            const blog = new Blob([blobParth], { type: event?.body?.type });
            this.downloadFile(blog);
          }
        },
      });
  }

  private downloadFile(blog: Blob) {
    const a = document.createElement('a');
    a.setAttribute('style', 'display:none;');
    document.body.appendChild(a);
    a.download = 'Employee_Certificate.pdf';
    a.href = URL.createObjectURL(blog);
    a.target = '_blank';
    a.click();
    document.body.removeChild(a);
  }
}
