import { Component, EventEmitter, Input, OnInit, Output } from '@angular/core';
import { User } from '@core/domain-classes/user';
import { MatCheckboxChange, MatCheckboxModule } from '@angular/material/checkbox';
import { Page } from '@core/domain-classes/page';
import { Action } from '@core/domain-classes/action';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { HasClaimDirective } from '@shared/has-claim.directive';
import { RouterModule } from '@angular/router';
import { TranslateModule } from '@ngx-translate/core';
import { MatCardModule } from '@angular/material/card';
import { MatButtonModule } from '@angular/material/button';
import { MatIconModule } from '@angular/material/icon';
import { FormsModule } from '@angular/forms';
import { BaseComponent } from '../../base.component';
import { MatExpansionModule } from "@angular/material/expansion";
import { Module } from '@core/domain-classes/module';

@Component({
  selector: 'app-user-permission-presentation',
  templateUrl: './user-permission-presentation.component.html',
  styleUrls: ['./user-permission-presentation.component.scss'],
  standalone: true,
  imports: [
    PageHelpTextComponent,
    HasClaimDirective,
    MatCheckboxModule,
    RouterModule,
    TranslateModule,
    MatCardModule,
    MatButtonModule,
    MatIconModule,
    FormsModule,
    MatExpansionModule,
  ]
})
export class UserPermissionPresentationComponent extends BaseComponent implements OnInit {
  @Input() modules: Module[] = [];
  @Input() user: User;
  @Output() manageUserClaimAction: EventEmitter<User> = new EventEmitter<User>();
  step: number = 0;

  constructor() {
    super();
  }

  ngOnInit(): void {
  }

  checkPermission(actionId: string): boolean {
    const pageAction = this.user?.userClaims?.find(c => c.actionId === actionId);
    if (pageAction) {
      return true;
    } else {
      return false;
    }
  }

  onPermissionChange(flag: any, page: Page, action: Action) {
    if (flag.checked) {
      this.user?.userClaims?.push({
        userId: this.user.id,
        claimType: action.code,
        claimValue: '',
        actionId: action.id,
        pageId: page.id
      })
    } else {
      const roleClaimToRemove = this.user?.userClaims?.find(c => c.actionId === action.id);
      if (roleClaimToRemove) {
        const index = this.user?.userClaims?.indexOf(roleClaimToRemove, 0);
        if (typeof index === 'number' && index > -1) {
          this.user?.userClaims?.splice(index, 1);
        }
      }
    }
  }

  onPageSelect(event: MatCheckboxChange, page: Page) {
    if (event.checked) {
      page.actions.forEach((action) => {
        if (!this.checkPermission(action.id)) {
          this.user.userClaims?.push({
            userId: this.user.id,
            claimType: action.code,
            claimValue: '',
            actionId: action.id,
            pageId: page.id
          });
        }
      });
    } else {
      var actions = page.actions?.map((c) => c.id);
      this.user.userClaims = this.user.userClaims?.filter(
        (c) => actions.indexOf(c.actionId) < 0
      );
    }
  }

  saveUserClaim() {
    this.manageUserClaimAction.emit(this.user);
  }

  selecetAll(event: MatCheckboxChange) {
    if (event.checked) {
      this.modules.forEach((module) => {
        module.pages.forEach((page) => {
          page.actions.forEach((action) => {
            if (!this.checkPermission(action.id)) {
              this.user.userClaims?.push({
                userId: this.user.id,
                claimType: action.code,
                claimValue: '',
                actionId: action.id,
                pageId: page.id
              });
            }
          });
        });
      });
    } else {
      this.user.userClaims = [];
    }
  }

  isModuleFullySelected(module: Module): boolean {
    if (!module.pages?.length) return false;
    for (const page of module.pages) {
      if (!page.actions?.length) continue;
      for (const action of page.actions) {
        if (!this.checkPermission(action.id)) {
          return false;
        }
      }
    }
    return true;
  }

  isModuleIndeterminate(module: Module): boolean {
    let total = 0;
    let selected = 0;
    if (!module.pages?.length) return false;
    for (const page of module.pages) {
      if (!page.actions?.length) continue;
      for (const action of page.actions) {
        total++;
        if (this.checkPermission(action.id)) {
          selected++;
        }
      }
    }
    return selected > 0 && selected < total;
  }

  onModuleSelect(event: MatCheckboxChange, module: Module) {
    if (event.checked) {
      module.pages?.forEach(page => {
        page.actions?.forEach(action => {
          if (!this.checkPermission(action.id)) {
            this.user.userClaims?.push({
              userId: this.user.id,
              claimType: action.code,
              claimValue: '',
              actionId: action.id,
              pageId: page.id
            });
          }
        });
      });
    } else {
      // Remove all actions in this module from userClaims
      const actionIds: string[] = [];
      module.pages?.forEach(page => {
        page.actions?.forEach(action => {
          actionIds.push(action.id);
        });
      });
      this.user.userClaims = this.user.userClaims?.filter(rc => !actionIds.includes(rc.actionId));
    }
  }

  isPageFullySelected(page: Page): boolean {
    if (!page.actions?.length) return false;
    return page.actions.every(action => this.checkPermission(action.id));
  }

  isPageIndeterminate(page: Page): boolean {
    if (!page.actions?.length) return false;
    const selected = page.actions.filter(action => this.checkPermission(action.id)).length;
    return selected > 0 && selected < page.actions.length;
  }
}
