import { AfterViewInit, Component, effect, inject, OnInit, ViewChild } from '@angular/core';
import {
  FormBuilder,
  FormGroup,
  Validators,
  FormArray,
  ReactiveFormsModule,
} from '@angular/forms';
import { Router, RouterModule } from '@angular/router';
import { MatButtonModule } from '@angular/material/button';
import { MatIconModule } from '@angular/material/icon';
import { MatSelectModule } from '@angular/material/select';
import { TranslateModule } from '@ngx-translate/core';
import { WorkflowStore } from '../../workflow-store';
import { WorkflowTransition } from '@core/domain-classes/workflow-transition';
import { NgxGraphModule, Node } from '@swimlane/ngx-graph';
import { fromToStepValidator } from './from-to-step-validate';
import { ClonerService } from '@core/services/clone.service';
import { FilterWorkflowStepPipe } from './filter-workflow-step.pipe';
import { MatCheckboxModule } from '@angular/material/checkbox';
import { atLeastOneRequiredValidator } from './at-least-one-required-validator';
import { MatButtonToggleModule } from '@angular/material/button-toggle';
import { PageHelpTextComponent } from '@shared/page-help-text/page-help-text.component';
import { MatCardModule } from '@angular/material/card';

@Component({
  selector: 'app-manage-transition',
  imports: [
    RouterModule,
    ReactiveFormsModule,
    TranslateModule,
    MatButtonModule,
    MatIconModule,
    MatSelectModule,
    NgxGraphModule,
    FilterWorkflowStepPipe,
    MatCheckboxModule,
    MatButtonToggleModule,
    PageHelpTextComponent,
    MatCardModule
  ],
  templateUrl: './manage-transition.component.html',
  styleUrls: ['./manage-transition.component.scss']
})
export class ManageTransitionComponent implements OnInit, AfterViewInit {
  private fb = inject(FormBuilder);
  public workflowStore = inject(WorkflowStore);
  private clonerService = inject(ClonerService);
  private router = inject(Router);

  transitionFormGroup: FormGroup;
  isLoading = false;
  currentStep = 2;
  nodes: Node[] = [];
  links: any[] = [];

  currentWorkflow = this.workflowStore.currentWorkflow();
  workflowSteps = this.currentWorkflow?.workflowSteps ?? [];
  workflowTransitions = this.currentWorkflow?.workflowTransitions ?? [];
  workflowInstances = this.currentWorkflow?.workflowInstances ?? [];
  @ViewChild('graph') graph: any;

  constructor() {
    effect(() => {
      const nextStep = this.workflowStore.currentStep();
      if (nextStep !== this.currentStep) {
        this.currentStep = nextStep;
        if (nextStep === 1) {
          this.onPreviousClick();
        } else {
          this.goToWorkflow();
        }
      }
    });
  }

  get transitions(): FormArray {
    return this.transitionFormGroup.get('transitions') as FormArray;
  }

  goToWorkflow() {
    this.router.navigate(['/workflow-settings']);
  }

  ngOnInit(): void {
    this.nodes = this.workflowSteps?.map((step: any) => ({
      id: step.id ?? '',
      label: step.stepName,
      data: step,
    }));

    this.transitionFormGroup = this.fb.group({
      transitions: this.fb.array([]),
    });

    if (this.workflowTransitions && this.workflowTransitions?.length === 0) {
      this.initialTransition();
    } else {
      for (let i = 0; i < this.workflowTransitions.length; i++) {
        this.addTransition(this.workflowTransitions[i]);
      }
      // this.workflowTransitions.forEach((transition) => this.addTransition(transition));
      this.addAllLink();
    }
  }
  ngAfterViewInit() {
    setTimeout(() => {
      if (this.graph) {
        this.graph?.zoomToFit(); // Auto zooms to fit view
      }
    });
  }

  checkUniqueTransitionName(index: number) {
    const transitionName = this.transitions.at(index).get('name')?.value;
    const isNotUnique = this.transitions.controls.some(
      (control, i) => i !== index && control.get('name')?.value === transitionName
    );

    if (isNotUnique) {
      this.transitions.at(index).get('name')?.setErrors({ notUnique: true });
    } else {
      const errors = this.transitions.at(index).get('name')?.errors;

      if (errors) {

        if (Object.keys(errors).length === 0) {
          this.transitions.at(index).get('name')?.setErrors(null);
        } else {
          this.transitions.at(index).get('name')?.setErrors(errors);
        }
      }
    }
  }

  filterSteps(index: number): void {
    this.removeFormItem(index);
    this.addAllLink();
  }

  removeFormItem(index: number): void {
    if (index == this.transitions.length - 1) {
      return;
    }
    const length = this.transitions.length;
    for (let i = index + 1; i < length; i++) {
      this.transitions.removeAt(index + 1);
    }

  }

  addAllLink() {
    const filerArray = this.transitions.getRawValue().filter((transition) => transition.fromStepId && transition.toStepId && transition.name);
    if (filerArray.length > 0) {
      this.links = this.clonerService.deepClone(filerArray.map((transition) => {
        return { source: transition.fromStepId, target: transition.toStepId, label: transition.name }
      }));
    }
  }

  initialTransition(): void {
    this.transitions.push(
      this.fb.group({
        id: [null],
        name: ['', Validators.required],
        fromStepId: ['', Validators.required],
        toStepId: ['', Validators.required],
        isFirstTransaction: [true],
        isUploadDocumentVersion: [false],
        isSignatureRequired: [false],
        workflowId: [
          null
        ],
        color: ['inprogress'],
        days: [0],
        hours: [0],
        minutes: [0],
        roleIds: [null],
        userIds: [null],
        orderNo: [0]
      }, { validators: [fromToStepValidator(), atLeastOneRequiredValidator()] }),
    );
  }
  onTransitionChange() {
    this.addAllLink();
  }

  getDays(): number[] {
    return Array.from({ length: 32 }, (_, i) => i);
  }

  getHours(): number[] {
    return Array.from({ length: 24 }, (_, i) => i);
  }

  getMinutes(): number[] {
    return Array.from({ length: 60 }, (_, i) => i);
  }

  addTransition(transition?: WorkflowTransition): void {
    if (this.transitions.valid) {
      this.transitions.push(
        this.fb.group({
          id: [transition?.id || null],
          name: [transition?.name || '', Validators.required],
          fromStepId: [{ value: transition?.fromStepId || '', disabled: transition?.id ? true : false }, [Validators.required]],
          toStepId: [{ value: transition?.toStepId || '', disabled: transition?.id ? true : false }, [Validators.required]],
          isFirstTransaction: [transition?.isFirstTransaction || false],
          isUploadDocumentVersion: [transition?.isUploadDocumentVersion || false],
          isSignatureRequired: [transition?.isSignatureRequired || false],
          workflowId: [this.currentWorkflow?.id || null,],
          days: [transition?.days || 0],
          hours: [transition?.hours || 0],
          minutes: [transition?.minutes || 0],
          color: [transition?.color || 'inprogress'],
          orderNo: [transition?.orderNo || 0],
          roleIds: [transition?.workflowTransitionRoles?.map((role) => role.roleId) || null],
          userIds: [transition?.workflowTransitionUsers?.map((user) => user.userId) || null],
        }, { validators: [fromToStepValidator(), atLeastOneRequiredValidator()] })
      );
    } else {
      this.transitionFormGroup.markAllAsTouched();
    }
  }

  removeTransition(index: number): void {
    this.removeFormItem(index);
    this.transitions.removeAt(index);
    if (this.transitions.length === 0) {
      this.initialTransition();
    }
    this.addAllLink();
  }

  saveTransition(): void {
    if (this.transitionFormGroup.invalid) {
      this.transitionFormGroup.markAllAsTouched();
      return;
    }

    this.isLoading = true;
    const transitionsData: WorkflowTransition[] = this.transitions.getRawValue().map((transition: any) => ({
      ...transition,
      workflowId: this.currentWorkflow?.id ?? '',
    }));
    
    if (transitionsData[0].id) {
      transitionsData[0].isFirstTransaction = true;
      this.workflowStore.updateWorkflowTransition(transitionsData);
    } else {
      this.workflowStore.addWorkflowTransition(transitionsData);
    }

  }
  onPreviousClick() {
    this.workflowStore.setCurrentStep(1);
    this.router.navigate(['/workflow-settings/manage/manage-steps']);
  }

}


